# labels VMD plugin
#
# Author: Jordi Cohen
#
# Implements a tape rule, a grid, or a scale indicator
# 
# usage: labels [tape|grid|off]

package provide labels 1.0

namespace eval ::labels:: {
  set labels_unit     1.  ;# base unit of scale
  set labels_logunit -1   ;# base unit of scale
  set labels_scale    1.  ;# actual scale of the top molecule
  set labels_mol     -1   ;# mol in which the drawing is done
  set labels_dirty    1   ;# recompute the labels?
    
  set display_width  1.  ;# max x in OpenGL
  set display_height 1.  ;# max y in OpenGL  
  set display_front  1.  ;# max z in OpenGL  
  
  set labels_color    black  ;# use a dark foreground color? 
  set labels_state    off
  
  set grid_on  0
  set scale_on 0
# LEGEND
  set legend_on 0
  set labels_on 0

  set scale_graphics_id     -1
# LEGEND
  set legend_graphics_id     -1
  set indicator_graphics_id -1
  
  #gui
  variable w
  
}


proc ::labels::setup_labels {} {
  variable labels_mol

  if ![catch {molinfo $labels_mol get name}] {return}
  
  set top [molinfo top]
  set labels_mol [mol new]
  mol rename $labels_mol "labels"
  if {$top >= 0} {
    mol top $top
    molinfo $labels_mol set scale_matrix [molinfo $top get scale_matrix]  
  }
  
  reset_colors
  
  trace add variable ::vmd_logfile write ::labels::logfile_cb
}


proc ::labels::remove_labels {} {
  variable labels_mol
  
  trace remove variable ::vmd_logfile write ::labels::logfile_cb

  catch {mol delete $labels_mol}
}


proc ::labels::reset_colors {} {
  variable labels_color
  
  if [display get backgroundgradient] {
    set backlight [eval vecadd [colorinfo rgb [color Display BackgroundBot]]]
  } else {
    set backlight [eval vecadd [colorinfo rgb [color Display Background]]]
  }
  if {$backlight <= 1.2} {
    set labels_color white
  } else {
    set labels_color black
  }
}


proc ::labels::redraw {} {
  variable labels_mol
  variable labels_scale
  variable labels_unit
  variable labels_logunit
  variable grid_on
  variable labels_on
  variable scale_on
# LEGEND
  variable legend_on
  variable display_height
  variable display_width
  variable display_front
        
  molinfo $labels_mol set center_matrix [list [transidentity]]
  molinfo $labels_mol set rotate_matrix [list [transidentity]]
  molinfo $labels_mol set global_matrix [list [transidentity]]
  molinfo $labels_mol set scale_matrix  [molinfo top get scale_matrix]
  
  if $::labels::labels_dirty {
    set labels_scale [lindex [molinfo $labels_mol get scale_matrix] 0 0 0]
    set logunit [expr ceil(-log10($labels_scale))-1]
    set labels_logunit $logunit
    set labels_unit [expr pow(10.,$labels_logunit)]

    set display_height [expr 0.25*[display get height]/$labels_scale]
    set display_width [expr $display_height*[lindex [display get size] 0]/[lindex [display get size] 1]]
    
    if [string equal [display get projection] "Orthographic"] {
      set display_front [expr (2.-[display get nearclip]-0.001)/$labels_scale]
    } else {
      set display_front 0.
    }
     
    graphics $labels_mol delete all
    graphics $labels_mol material Opaque
    
    if $grid_on  {draw_grid}
    if $scale_on {draw_scale}
# LEGEND
    if $legend_on {draw_legend}
    if $labels_on {draw_labels}
 
    if {$grid_on || $labels_on} {draw_indicator}
    
    set ::labels::labels_dirty 0
  }
}


proc ::labels::draw_grid {} {
  variable labels_mol
  variable labels_unit
  variable display_height
  variable display_width
  variable display_front

  set maxx $display_width  
  set minx [expr -ceil($display_width/(10.*$labels_unit))*10.*$labels_unit] 
  set maxy $display_height  
  set miny [expr -ceil($display_height/(10.*$labels_unit))*10.*$labels_unit] 
 
  graphics $labels_mol color gray
  #draw material Transparent
  for {set tick $minx} {$tick <= $maxx} {set tick [expr $tick + $labels_unit]} {
    graphics $labels_mol  line "$tick $miny 0." "$tick $maxy 0." width 1 style dashed
  }
  for {set tick $miny} {$tick <= $maxy} {set tick [expr $tick + $labels_unit]} {
    graphics $labels_mol  line "$minx $tick 0." "$maxx $tick 0." width 1 style dashed
  }
 
  #draw material Opaque
  for {set tick $minx} {$tick <= $maxx} {set tick [expr $tick + 10.*$labels_unit]} {
    graphics $labels_mol line "$tick $miny 0." "$tick $maxy 0." width 2
  }
  for {set tick $miny} {$tick <= $maxy} {set tick [expr $tick + 10.*$labels_unit]} {
    graphics $labels_mol  line "$minx $tick 0." "$maxx $tick 0." width 2
  }
}


proc ::labels::draw_indicator {} {
  variable labels_mol
  variable labels_unit
  variable display_height
  variable display_width
  variable display_front
  
  graphics $labels_mol color gray
  graphics $labels_mol text "[expr -0.99*$display_width] [expr -0.97*$display_height] $display_front" "[format "%g" $labels_unit]A" size 0.8
}


proc ::labels::draw_scale {} {
  variable labels_mol
  variable labels_unit
  variable display_height
  variable display_width
  variable display_front  

  graphics $labels_mol color $::labels::labels_color
  graphics $labels_mol text "[expr -0.9*$display_width] [expr -0.95*$display_height] $display_front" "[format "%g" $labels_unit] A" size 1.0
  graphics $labels_mol line "[expr -0.9*$display_width] [expr -0.9*$display_height] $display_front" "[expr -0.9*$display_width+$labels_unit] [expr -0.9*$display_height] $display_front" width 10
}

##
# Legende der Molekülfarben
##

proc ::labels::draw_legend {} {
  variable labels_mol
  variable labels_unit
  variable display_height
  variable display_width
  variable display_front  


set fp [open "ProteinEigenschaften.csv" r]
    set file_data1 [read $fp]
    close $fp

set data1 [split $file_data1 "\n"]

#list with the radii sort by order in .bngl & .geo files
#set radlist {{0} {22.03} {24.46} {28.88} {19.21} {19.70} {16.51} {19.51} {19.00} {15.33} {16.51} {23.88} {13.10} {20.71} {26.95} {15.33} {14.66} {14.66} {15.00} {13.53} {20.63} {12.65} {18.58}}

#set protnamelst [list "bla" "CENP_U" "CENP_B" "CENP_C" "CENP_K" "CENP_O" "CENP_R" "CENP_P" "CENP_Q" "CENP_A" "CENP_M" "CENP_T" "CENP_W" "CENP_N" "CENP_I" "CENP_S" "H2A" "H2B" "H3" "H4" "CENP_L" "CENP_X" "CenpH"]



for {set i 1} { 63 >= $i } {incr i} {
        	graphics $labels_mol color $i
        	graphics $labels_mol sphere "[expr 0.8*$display_width] [expr 0.9*$display_height-[expr $i-1]*0.028*$display_height] $display_front" radius [expr 0.3 * [lindex $data1 $i 2]] resolution 50
        	graphics $labels_mol text "[expr 0.82*$display_width] [expr 0.9*$display_height-[expr $i-1]*0.028*$display_height] $display_front" "[lindex $data1 $i 1]($i)" size 0.5
#        graphics $labels_mol sphere "[expr 0.75*$display_width] [expr 0.9*$display_height-[expr $i-1]*0.085*$display_height] $display_front" radius [expr 0.4 * [lindex $data1 $i 2]] resolution 50
#        graphics $labels_mol text "[expr 0.78*$display_width] [expr 0.9*$display_height-[expr $i-1]*0.085*$display_height] $display_front" "[lindex $data1 $i 1]($i)" size 0.85
}





  
#  graphics $labels_mol sphere "[expr -0.9*$display_width] [expr 0.9*$display_height] $display_front" radius 5.4 resolution 50
#  graphics $labels_mol text "[expr -0.85*$display_width] [expr 0.9*$display_height] $display_front" " CENP_A" size 1.0
}


proc ::labels::draw_labels {} {
  variable labels_mol
  variable labels_unit
  variable labels_scale
  variable display_height
  variable display_width
  variable display_front
  
  set pixelwidth [expr 2.*$display_height/[lindex [display get size] 0]]
     
  set edgewidth [expr 0.01666*[display get height]/$labels_scale]
  set edge_x  [expr -$display_width + $edgewidth]
  set edge_xp [expr $edge_x + $pixelwidth]
  set edge_x1 [expr -$display_width + 0.1*$edgewidth]
  set edge_x2 [expr -$display_width + 0.6*$edgewidth]
  set edge_y  [expr -$display_height + $edgewidth]
  set edge_yp [expr $edge_y + $pixelwidth]
  set edge_y1 [expr -$display_height + 0.1*$edgewidth]
  set edge_y2 [expr -$display_height + 0.6*$edgewidth]

  set maxx [expr ceil($display_width/(10.*$labels_unit))*10.*$labels_unit]  
  set minx $edge_x
  set maxy [expr ceil($display_height/(10.*$labels_unit))*10.*$labels_unit]
  set miny $edge_y

  graphics $labels_mol color yellow

  graphics $labels_mol  triangle "-$display_width $display_height $display_front" "$edge_x $display_height $display_front" "-$display_width -$display_height $display_front"
  graphics $labels_mol  triangle "$edge_x $display_height $display_front" "$edge_x -$display_height $display_front" "-$display_width -$display_height $display_front"
  graphics $labels_mol  triangle "-$display_width $edge_y $display_front" "$display_width $edge_y $display_front" "-$display_width -$display_height $display_front"
  graphics $labels_mol  triangle "-$display_width -$display_height $display_front" "$display_width $edge_y $display_front" "$display_width -$display_height $display_front"

  graphics $labels_mol color white
  graphics $labels_mol line "$edge_x $display_height $display_front" "$edge_x $edge_y $display_front" width 3
  graphics $labels_mol line "$edge_x $edge_y $display_front" "$display_width $edge_y $display_front" width 3
  
  graphics $labels_mol color black
  graphics $labels_mol line "$edge_xp $display_height $display_front" "$edge_xp $edge_yp $display_front" width 1
  graphics $labels_mol line "$edge_xp $edge_yp $display_front" "$display_width $edge_yp $display_front" width 1
    
  graphics $labels_mol color black        
  for {set tick $maxy} {$tick >= $miny} {set tick [expr $tick - 10.*$labels_unit]} {
    graphics $labels_mol line "$edge_x1 $tick $display_front" "$edge_xp $tick $display_front" width 2
  }
  for {set tick $maxy} {$tick >= $miny} {set tick [expr $tick - 1.*$labels_unit]} {
    graphics $labels_mol line "$edge_x2 $tick $display_front" "$edge_xp $tick $display_front" width 1
  }
  
  for {set tick $maxx} {$tick >= $minx} {set tick [expr $tick - 10.*$labels_unit]} {
    graphics $labels_mol line "$tick $edge_y1 $display_front" "$tick $edge_yp $display_front" width 2
  }
  for {set tick $maxx} {$tick >= $minx} {set tick [expr $tick - 1.*$labels_unit]} {
    graphics $labels_mol line "$tick $edge_y2 $display_front" "$tick $edge_yp $display_front" width 1
  }
}


proc ::labels::logfile_cb { args } {
  # Check for display transforms
  if {[string match "rotate *" $::vmd_logfile] || [string match "translate *" $::vmd_logfile]} { 
    redraw
  } elseif {[string match "scale *" $::vmd_logfile] || [string match "mol top *" $::vmd_logfile]} {
    set ::labels::labels_dirty 1 
    redraw
  } elseif {[string match "color *" $::vmd_logfile] || [string match "display *" $::vmd_logfile]} {
    set ::labels::labels_dirty 1 
    reset_colors
    redraw
  } 
}



proc ::labels::show {args} {
  variable labels_mol
  variable grid_on
  variable labels_on
  variable scale_on
# LEGEND
  variable legend_on
  variable labels_state
  
  set overlay [lindex $args 0]
  set state [string is true [lindex $args 1]]
  
  if {"$overlay" == "grid"} {
    set labels_on 0
    set scale_on 0
#LEGEND
    set legend_on 0
    set grid_on $state
    set labels_state $overlay
  } elseif {"$overlay" == "tape"} {
    set labels_on 1
    set scale_on 0
#LEGEND
    set legend_on 0
    set grid_on  0
    set labels_state $overlay
  } elseif {"$overlay" == "scale"} {
    set labels_on 0
    set scale_on 1
#LEGEND
    set legend_on 0
    set grid_on  0
    set labels_state $overlay
  } elseif {"$overlay" == "legend"} {
    set labels_on 0
    set scale_on 0
#LEGEND
    set legend_on 1
    set grid_on  0
    set labels_state $overlay
  } elseif {"$overlay" == "off"} {
    set labels_on 0
    set scale_on 0
#LEGEND
    set legend_on 0
    set grid_on  0
    set labels_state $overlay
  } else {
    puts "usage: labels \[tape|grid|scale|legend|off\]"
    puts "Displays a dynamic labels for the top molecule."
  }
  
  if {$grid_on || $labels_on || $scale_on || $legend_on} {
    setup_labels
    set ::labels::labels_dirty 1
    redraw
  }
  if {!$grid_on && !$labels_on && !$scale_on && !$legend_on} {
    remove_labels
  } 
}


proc labels {args} {
  eval ::labels::show $args
}



###################################################################
###                            GUI 
###################################################################


proc labels_tk {} {
  return [labels::labels_gui]
}

proc labels::labels_gui {} {
  variable w
  variable labels_state

  # If already initialized, just turn on
  if [winfo exists .labels] {
    wm deiconify .labels
    raise .labels
    return $w
  }

  # Initialize window
  set w [toplevel .labels]
  wm title $w "labels Settings" 
  wm resizable $w 0 0


  # Status Frame
  set frame [frame $w.pick]

  label $frame.descr -text "Pick a labels style:" -justify left
  
  frame $frame.choose
  set menu [tk_optionMenu $frame.choose.menu labels::labels_state ""]
  $menu delete 0 
  $menu insert 0 radiobutton -label off -command {labels off}
  $menu insert 1 radiobutton -label tape -command {labels tape}
  $menu insert 2 radiobutton -label grid -command {labels grid}
  $menu insert 3 radiobutton -label scale -command {labels scale}
# LEGEND
  $menu insert 3 radiobutton -label legend -command {labels legend}
  pack $frame.choose.menu -expand yes -fill x 
  pack $frame.descr $frame.choose -side left -expand 1 -padx 4 -pady 4
  pack $frame
  
  button $w.ortho -text "Switch to Orthographic Mode" -command "display projection orthographic"
  pack  $w.ortho -padx 4 -pady 4
  
  return $w
}
